//
//  TAWEditEmailViewController.swift
//  TAW
//
//  Created by Andrew Steven on 14/01/2019.
//  Copyright © 2019 PixelBeard. All rights reserved.
//

import UIKit

class TAWEditEmailViewController: PBAnimatedViewController {
    
    // MARK: - Interface outlets -
    
    @IBOutlet weak var backButton: UIButton!
    @IBOutlet weak var emailTextField: PBTextField!
    @IBOutlet weak var saveButton: PBButton!
    
    @IBOutlet weak var saveButtonHeight: NSLayoutConstraint!
    
    // MARK: - Life cycle -

    override func viewDidLoad() {
        super.viewDidLoad()

        emailTextField.theTextFieldLabel.text = "EMAIL"
        emailTextField.theTextField.text = LocalStorage.shared.user?.email
        emailTextField.setType(type: .email, validation_type: .email)
        emailTextField.theTextField.addTarget(self,
                                              action: #selector(textFieldDidChange(textField:)),
                                              for: .editingChanged)
        
        let tapGesture = UITapGestureRecognizer(target: self,
                                                action: #selector(dismissKeyboard))
        tapGesture.delegate = self
        self.view.addGestureRecognizer(tapGesture)
        
        self.saveButton.buttonBackgroundColor = TAW.Colors.greyButtonColor
        self.saveButton.shaddowOppacity = 0
    }
    
    override func viewWillAppear(_ animated: Bool) {
        self.animations = [.slide(.left, .slightly), .fadeIn]
        super.viewWillAppear(animated)
    }

    override func viewDidAppear(_ animated: Bool) {
        super.viewDidAppear(animated)
        self.setupView()
    }
    
    private func setupView() {
        if UIDevice.isSmallDevice {
            self.view.layoutIfNeeded()
            self.saveButtonHeight.constant = 45
            self.view.layoutIfNeeded()
        }

        if Connectivity.isConnectedToInternet {
            emailTextField.theTextField.isEnabled = true
            saveButton.isEnabled = true
        } else {
            emailTextField.theTextField.isEnabled = false
            emailTextField.alpha = 0.3
            saveButton.isEnabled = false
            PBBannerView.shared.showBanner(inView: self.navigationController!.view, withTitle: "No internet connection", style: .noInternet)
        }
    }
    
    // MARK: - UITextField -
    
    @objc func textFieldDidChange(textField: UITextField) {
        self.invisibleCheck()
    }
    
    @objc func dismissKeyboard() {
        self.view.endEditing(true)
    }
    
    // MARK: - Button actions -
    
    @IBAction func goBack() {
        self.navigationController?.popViewController(animated: true)
    }
    
    @IBAction func save() {
        if valid() {
            self.checkEmailIsAvailable()
        }
    }
    
    // MARK: - Validation functions -
    
    private func valid() -> Bool {
        let validEmail = self.emailTextField.checkValidation(required: true, type: .email)
        if !validEmail { return false }
        return true
    }
    
    private func invisibleCheck() {
        
        var isEnabled = false
        let validEmail = self.emailTextField.invisibleCheckValidation(type: .email)
        
        if validEmail && emailTextField.theTextField.text != LocalStorage.shared.user?.email {
            isEnabled = true
        }
        
        if isEnabled {
            self.saveButton.buttonBackgroundColor = TAW.Colors.redButtonColor
            self.saveButton.shaddowOppacity = 1
        } else {
            self.saveButton.buttonBackgroundColor = TAW.Colors.greyButtonColor
            self.saveButton.shaddowOppacity = 0
        }
    }
    
    // MARK: - API functions - 
    
    private func checkEmailIsAvailable() {
        
        let params = [
            "email": self.emailTextField.theTextField.text!
        ]
        
        APIClient.checkEmailIsAvailable(parameters: params) { (jsonResponse, error) in
            if jsonResponse["success"] as? Bool == true {
                self.updateEmail()
            } else {
                self.emailTextField.overrideFailValidation(message: "Email already registered")
            }
        }
    }
    
    private func updateEmail() {
        
        PBLoadingOverlay.shared.showOverlay(view: (self.navigationController?.view)!,
                                            withTitle: "Updating Email")
        
        let user = LocalStorage.shared.user!
        let params = [
            "old_email": user.email!,
            "new_email": self.emailTextField.theTextField.text!
        ]
        
        APIClient.updateUserEmail(parameters: params) { (jsonResponse, error) in
            if jsonResponse["success"] as? Bool == true {
                let user = LocalStorage.shared.user
                user?.email = self.emailTextField.theTextField.text!
                LocalStorage.shared.user = user
                
                PBLoadingOverlay.shared.hideOverlay {
                    self.navigationController?.popViewController(animated: true)
                }
            } else {
                PBLoadingOverlay.shared.hideOverlay {
                    let message = jsonResponse["message"] as! String
                    if message == "Invalid Token" {
                        self.invalidateSession()
                    } else {
                        self.showAlertForAPIError(jsonResponse["message"] as! String)
                    }
                }
            }
        }
    }
}
